package org.hl7.fhir.validation.cli.tasks;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;

import org.hl7.fhir.utilities.TimeTracker;
import org.hl7.fhir.utilities.VersionUtilities;
import org.hl7.fhir.utilities.filesystem.ManagedFileAccess;
import org.hl7.fhir.utilities.npm.CommonPackages;
import org.hl7.fhir.validation.ValidationEngine;
import org.hl7.fhir.validation.cli.model.CliContext;
import org.hl7.fhir.validation.cli.services.ComparisonService;
import org.hl7.fhir.validation.cli.services.ValidationService;
import org.hl7.fhir.validation.cli.utils.Display;
import org.hl7.fhir.validation.cli.utils.Params;

public class CompareTask extends ValidationEngineTask {
  @Override
  public String getName() {
    return "compare";
  }

  @Override
  public String getDisplayName() {
    return "Comparing Profiles";
  }

  @Override
  public boolean isHidden() {
    return false;
  }

  @Override
  public boolean shouldExecuteTask(CliContext cliContext, String[] args) {
    return Params.hasParam(args, Params.COMPARE);
  }

  @Override
  public void printHelp(PrintStream out) {
    Display.displayHelpDetails(out,"help/compare.txt");
  }

  @Override
  public void executeTask(ValidationService validationService, ValidationEngine validationEngine, CliContext cliContext, String[] args, TimeTracker tt, TimeTracker.Session tts) throws Exception {
    Display.printCliParamsAndInfo(args);
    if (!destinationDirectoryValid(Params.getParam(args, Params.DESTINATION))) {
      return;
    }
    if (cliContext.getSv() == null) {
      cliContext.setSv(validationService.determineVersion(cliContext));
    }
    String v = VersionUtilities.getCurrentVersion(cliContext.getSv());
    String definitions = VersionUtilities.packageForVersion(v) + "#" + v;
    ValidationEngine validator = validationService.initializeValidator(cliContext, definitions, tt);
    validator.loadPackage(CommonPackages.ID_PUBPACK, null);
    ComparisonService.doLeftRightComparison(args, Params.getParam(args, Params.DESTINATION), validator);
  }

  private boolean destinationDirectoryValid(String dest) throws IOException {
    if (dest == null) {
      System.out.println("no -dest parameter provided");
      return false;
    } else if (!ManagedFileAccess.file(dest).isDirectory()) {
      System.out.println("Specified destination (-dest parameter) is not valid: \"" + dest + "\")");
      return false;
    } else {
      System.out.println("Valid destination directory provided: \"" + dest + "\")");
      return true;
    }
  }
}
