/*
 * Decompiled with CFR 0.152.
 */
package ca.uhn.fhir.jpa.migrate;

import ca.uhn.fhir.i18n.Msg;
import ca.uhn.fhir.jpa.migrate.DriverTypeEnum;
import ca.uhn.fhir.jpa.migrate.HapiMigrationException;
import ca.uhn.fhir.jpa.migrate.HapiMigrationLock;
import ca.uhn.fhir.jpa.migrate.HapiMigrationStorageSvc;
import ca.uhn.fhir.jpa.migrate.IHapiMigrationCallback;
import ca.uhn.fhir.jpa.migrate.MigrationResult;
import ca.uhn.fhir.jpa.migrate.MigrationTaskList;
import ca.uhn.fhir.jpa.migrate.dao.HapiMigrationDao;
import ca.uhn.fhir.jpa.migrate.taskdef.BaseTask;
import ca.uhn.fhir.jpa.migrate.taskdef.InitializeSchemaTask;
import ca.uhn.fhir.jpa.migrate.tasks.api.TaskFlagEnum;
import ca.uhn.fhir.system.HapiSystemProperties;
import ca.uhn.fhir.util.StopWatch;
import com.google.common.annotations.VisibleForTesting;
import jakarta.annotation.Nonnull;
import java.sql.SQLException;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import javax.sql.DataSource;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HapiMigrator {
    private static final Logger ourLog = LoggerFactory.getLogger(HapiMigrator.class);
    private final MigrationTaskList myTaskList = new MigrationTaskList();
    private boolean myDryRun;
    private boolean myRunHeavyweightSkippableTasks;
    private boolean myNoColumnShrink;
    private final DriverTypeEnum myDriverType;
    private final DataSource myDataSource;
    private final HapiMigrationStorageSvc myHapiMigrationStorageSvc;
    private List<IHapiMigrationCallback> myCallbacks = Collections.emptyList();

    public HapiMigrator(String theMigrationTableName, DataSource theDataSource, DriverTypeEnum theDriverType) {
        this.myDriverType = theDriverType;
        this.myDataSource = theDataSource;
        this.myHapiMigrationStorageSvc = new HapiMigrationStorageSvc(new HapiMigrationDao(theDataSource, theDriverType, theMigrationTableName));
    }

    public DataSource getDataSource() {
        return this.myDataSource;
    }

    public boolean isDryRun() {
        return this.myDryRun;
    }

    public void setDryRun(boolean theDryRun) {
        this.myDryRun = theDryRun;
    }

    public boolean isRunHeavyweightSkippableTasks() {
        return this.myRunHeavyweightSkippableTasks;
    }

    public void setRunHeavyweightSkippableTasks(boolean theRunHeavyweightSkippableTasks) {
        this.myRunHeavyweightSkippableTasks = theRunHeavyweightSkippableTasks;
    }

    public boolean isNoColumnShrink() {
        return this.myNoColumnShrink;
    }

    public void setNoColumnShrink(boolean theNoColumnShrink) {
        this.myNoColumnShrink = theNoColumnShrink;
    }

    public DriverTypeEnum getDriverType() {
        return this.myDriverType;
    }

    protected StringBuilder buildExecutedStatementsString(MigrationResult theMigrationResult) {
        StringBuilder statementBuilder = new StringBuilder();
        String lastTable = null;
        for (BaseTask.ExecutedStatement next : theMigrationResult.executedStatements) {
            if (!Objects.equals(lastTable, next.getTableName())) {
                statementBuilder.append("\n\n-- Table: ").append(next.getTableName()).append("\n");
                lastTable = next.getTableName();
            }
            statementBuilder.append(next.getSql()).append(";\n");
            for (Object nextArg : next.getArguments()) {
                statementBuilder.append("  -- Arg: ").append(nextArg).append("\n");
            }
        }
        return statementBuilder;
    }

    public void clearMigrationLockWithUUID(String theUUID) {
        ourLog.info("Attempting to remove lock entry. [uuid={}]", (Object)theUUID);
        boolean success = this.myHapiMigrationStorageSvc.deleteLockRecord(theUUID);
        if (success) {
            ourLog.info("Successfully removed lock entry. [uuid={}]", (Object)theUUID);
        } else {
            ourLog.error("Did not successfully remove lock entry. [uuid={}]", (Object)theUUID);
        }
    }

    public MigrationResult migrate() {
        ourLog.info("Loaded {} migration tasks", (Object)this.myTaskList.size());
        MigrationResult retval = new MigrationResult();
        try (HapiMigrationLock ignored = new HapiMigrationLock(this.myHapiMigrationStorageSvc);){
            MigrationTaskList newTaskList = this.myHapiMigrationStorageSvc.diff(this.myTaskList);
            ourLog.info("{} of these {} migration tasks are new.  Executing them now.", (Object)newTaskList.size(), (Object)this.myTaskList.size());
            try (DriverTypeEnum.ConnectionProperties connectionProperties = this.getDriverType().newConnectionProperties(this.getDataSource());){
                if (!this.isRunHeavyweightSkippableTasks()) {
                    newTaskList.removeIf(BaseTask::isHeavyweightSkippableTask);
                }
                boolean initializedSchema = false;
                int skippedTasksDueToSchemaMigration = 0;
                for (BaseTask next : newTaskList) {
                    if (initializedSchema && !next.hasFlag(TaskFlagEnum.RUN_DURING_SCHEMA_INITIALIZATION)) {
                        ourLog.debug("Skipping task {} because schema is being initialized", (Object)next.getMigrationVersion());
                        this.recordTaskAsCompletedIfNotDryRun(next, 0L, true);
                        ++skippedTasksDueToSchemaMigration;
                        continue;
                    }
                    next.setDriverType(this.getDriverType());
                    next.setDryRun(this.isDryRun());
                    next.setNoColumnShrink(this.isNoColumnShrink());
                    next.setConnectionProperties(connectionProperties);
                    this.executeTask(next, retval);
                    initializedSchema |= next.initializedSchema();
                }
                if (skippedTasksDueToSchemaMigration > 0) {
                    ourLog.info("Skipped {} migration tasks because schema is being initialized", (Object)skippedTasksDueToSchemaMigration);
                }
            }
        }
        catch (Exception e) {
            ourLog.error("Migration failed", (Throwable)e);
            throw e;
        }
        ourLog.info(retval.summary());
        if (this.isDryRun()) {
            StringBuilder statementBuilder = this.buildExecutedStatementsString(retval);
            ourLog.info("SQL that would be executed:\n\n***********************************\n{}***********************************", (Object)statementBuilder);
        }
        return retval;
    }

    private void executeTask(BaseTask theTask, MigrationResult theMigrationResult) {
        StopWatch sw = new StopWatch();
        try {
            if (this.isDryRun()) {
                ourLog.info("Dry run {} {}", (Object)theTask.getMigrationVersion(), (Object)theTask.getDescription());
            } else {
                ourLog.info("Executing {} {}", (Object)theTask.getMigrationVersion(), (Object)theTask.getDescription());
            }
            this.preExecute(theTask);
            theTask.execute();
            this.recordTaskAsCompletedIfNotDryRun(theTask, sw.getMillis(), true);
            theMigrationResult.changes += theTask.getChangesCount();
            theMigrationResult.executionResult = theTask.getExecutionResult();
            theMigrationResult.executedStatements.addAll(theTask.getExecutedStatements());
            theMigrationResult.succeededTasks.add(theTask);
        }
        catch (HapiMigrationException | SQLException e) {
            theMigrationResult.failedTasks.add(theTask);
            this.recordTaskAsCompletedIfNotDryRun(theTask, sw.getMillis(), false);
            String description = theTask.getDescription();
            if (StringUtils.isBlank((CharSequence)description)) {
                description = theTask.getClass().getSimpleName();
            }
            String prefix = String.format("Failure executing task '%s', for driver: %s, aborting! Cause: ", new Object[]{description, this.getDriverType()});
            throw new HapiMigrationException(Msg.code((int)47) + prefix + String.valueOf(e), theMigrationResult, e);
        }
    }

    private void preExecute(BaseTask theTask) {
        this.myCallbacks.forEach(action -> action.preExecution(theTask));
    }

    private void recordTaskAsCompletedIfNotDryRun(BaseTask theNext, long theExecutionMillis, boolean theSuccess) {
        if (!theNext.isDryRun()) {
            this.myHapiMigrationStorageSvc.saveTask(theNext, Math.toIntExact(theExecutionMillis), theSuccess);
        }
    }

    public void addTasks(Iterable<BaseTask> theMigrationTasks) {
        if (HapiSystemProperties.isUnitTestModeEnabled()) {
            for (BaseTask task : theMigrationTasks) {
                if (!(task instanceof InitializeSchemaTask)) continue;
                this.addTask(task);
            }
        } else {
            this.myTaskList.append(theMigrationTasks);
        }
    }

    public void addAllTasksForUnitTest(Iterable<BaseTask> theMigrationTasks) {
        this.myTaskList.append(theMigrationTasks);
    }

    public void addTask(BaseTask theTask) {
        this.myTaskList.add(theTask);
    }

    public void setCallbacks(@Nonnull List<IHapiMigrationCallback> theCallbacks) {
        Validate.notNull(theCallbacks, (String)"theCallbacks must not be null", (Object[])new Object[0]);
        this.myCallbacks = theCallbacks;
    }

    @VisibleForTesting
    public void removeAllTasksForUnitTest() {
        this.myTaskList.clear();
    }

    public void createMigrationTableIfRequired() {
        if (!this.myDryRun) {
            this.myHapiMigrationStorageSvc.createMigrationTableIfRequired();
        }
    }
}

